/*
 * Demo ET-ESP32-RS485 Hardware Board
 * MCU      : ESP32-WROVER
 *          : Arduino Tools Board : ESP32 Wrover Module
 * Software : https://dl.espressif.com/dl/package_esp32_index.json        
 * Reserve  : Boot Config       
 *          : -> IO0(BOOT)
 *          : -> IO2(Don't Card on 3V3V Version)
 *          : -> IO5
 *          : -> IO12(MTDI)
 *          : -> IO15(MTDO)
 * Reserve  : SFLASH        
 *          : -> CMD
 *          : -> CLK
 *          : -> SD0
 *          : -> SD1
 *          : -> SD2
 *          : -> SD3
 * Debug    : Serial0 : USART0(USB)        
 *          : -> RX0(Debug & Download:IO3)
 *          : -> TX0(Debug & Download:IO1)
 * NB-IoT   : Serial1 : SIM7020E(BK-7020 V2)       
 *          : -> RX1(T:IO14)
 *          : -> TX1(R:IO13)
 *          : -> PWRKEY(K:IO33)
 *          : -> SLEEP(S:IO32)
 * RS485    : Serial1 : RS485  
 *          : -> RX2(IO26)
 *          : -> TX2(IO27)
 *          : -> DIR(Direction : IO25 : LOW = Receive, HIGH = Send)
 * I2C#1    : I2C BUS
 *          : -> SCL1(IO22)
 *          : -> SDA1(IO21)
 * RTC      : -> RTC:DS3231/DS3232
 *          : -> PCF8574/A(Relay8)
 *          : -> PCF8574/A(DC-IN8)
 * LED      : LED Status(Active High)
 *          : -> LED0(IO2)
 * Demo     : Interface -> MODBUS RTU RELAY2
 *          : -> Setup MODBUS RTU RELAY2 ID = 1(SW[1:ON],[5:ON] 
 *          : ET-ESP32-RS485 : Setup RS422/485 Hardware
 *          : -> FULL/HALF   = HALF
 *          : -> 422/485     = 485
 *          : -> 4WIRE/2WIRE = 2WIRE
 *          : -> RZ(ENA/DIS) = ENA
 *          : -> RH(ENA/DIS) = ENA
 *          : -> RL(ENA/DIS) = ENA
 */


//=================================================================================================
#include <Wire.h> 
//=================================================================================================
#include "ETT_ModbusRTU.h"
//=================================================================================================

//=================================================================================================
// Start of Default Hardware : ET-ESP32-RS485
//=================================================================================================
// Remap Pin USART -> C:\Users\Admin\Documents\Arduino\hardware\espressif\esp32\cores\esp32\HardwareSerial.cpp
//                    C:\Users\Admin\AppData\Local\Arduino15\packages\esp32\hardware\esp32\1.0.0\cores\esp32\HardwareSerial.cpp
//=================================================================================================
#include <HardwareSerial.h>
//=================================================================================================
#define SerialDebug  Serial                                                                       // USB Serial(Serial0)
//=================================================================================================
#define SerialNBIOT_RX_PIN    14
#define SerialNBIOT_TX_PIN    13
#define SerialNBIOT  Serial1                                                                      // Serial1(IO13=TXD,IO14=RXD)
//=================================================================================================
#define SerialRS485_RX_PIN    26
#define SerialRS485_TX_PIN    27
#define SerialRS485  Serial2                                                                      // Serial2(IO27=TXD,IO26=RXD)
//=================================================================================================
#define RS485_DIRECTION_PIN   25                                                                  // ESP32-WROVER :IO25
#define RS485_RXD_SELECT      LOW
#define RS485_TXD_SELECT      HIGH
//=================================================================================================
#define SIM7020E_PWRKEY_PIN   33                                                                  // ESP32-WROVER :IO33
#define SIM7020E_SLEEP_PIN    32                                                                  // ESP32-WROVER :IO32
#define SIM7020E_PWRKEY_LOW   LOW                                                                 // Start Power-ON
#define SIM7020E_PWRKEY_HIGH  HIGH                                                                // Release Signal
#define SIM7020E_SLEEP_LOW    LOW                                                                 // Pull-Up DTR to Enable Sleep
#define SIM7020E_SLEEP_HIGH   HIGH                                                                // DTR=Low(Wakeup From Sleep)
//=================================================================================================
#define I2C_SCL1_PIN          22                                                                  // ESP32-WROVER : IO22(SCL1)
#define I2C_SDA1_PIN          21                                                                  // ESP32-WROVER : IO21(SDA1)
//=================================================================================================

//=================================================================================================
#define LED_PIN               2                                                                   // ESP-WROVER  : IO2
#define LedON                 1
#define LedOFF                0
//=================================================================================================
#define CS_SD_CARD_PIN        4                                                                   // ESP-WROVER  : IO4
#define SD_CARD_DISABLE       1
#define SD_CARD_ENABLE        0
//=================================================================================================
// End of Default Hardware : ET-ESP32-RS485
//=================================================================================================

//=================================================================================================
// data array for modbus network sharing
//=================================================================================================
uint16_t au16dataRelay[2];                                                                       // data array for modbus network sharing
uint8_t u8state;                                                                                 // machine state
uint8_t u8query;                                                                                 // pointer to message query
//=================================================================================================

/**
 *  Modbus object declaration
 *  u8id : node id = 0 for master, = 1..247 for slave
 *  u8serno : serial port (use 0 for Serial)
 *  u8txenpin : 0 for RS-232 and USB-FTDI 
 *               or any pin number > 1 for RS-485
 */
Modbus master(0,                                                                                 // node id = 0(master)
              SerialRS485,                                                                       // Serial(2)
              RS485_DIRECTION_PIN);                                                              // RS485 Modbus

/**
 * This is an structe which contains a query to an slave device
 */
modbus_t telegram[2];                                                                            // 2-Modbus Commans
unsigned long u32wait;
//=================================================================================================

void setup() 
{
  //===============================================================================================
  // Start of Initial Default Hardware : ET-ESP32-RS485
  //===============================================================================================
  pinMode(LED_PIN, OUTPUT);
  digitalWrite(LED_PIN, LedOFF);
  //===============================================================================================
  pinMode(CS_SD_CARD_PIN, OUTPUT);
  digitalWrite(CS_SD_CARD_PIN, SD_CARD_DISABLE);
  //===============================================================================================
  Wire.begin(I2C_SDA1_PIN,I2C_SCL1_PIN);                                                      
  //===============================================================================================
  pinMode(RS485_DIRECTION_PIN, OUTPUT);                                                          // RS485 Direction
  digitalWrite(RS485_DIRECTION_PIN, RS485_RXD_SELECT);
  //===============================================================================================
  SerialDebug.begin(115200);
  while(!SerialDebug);
  //===============================================================================================
  SerialNBIOT.begin(115200, SERIAL_8N1, SerialNBIOT_RX_PIN, SerialNBIOT_TX_PIN);
  while(!SerialNBIOT);
  //===============================================================================================
  SerialRS485.begin(9600, SERIAL_8N1, SerialRS485_RX_PIN, SerialRS485_TX_PIN);
  while(!SerialRS485);
  //===============================================================================================
  // End of Initial Default Hardware : ESP32
  //===============================================================================================

  //===============================================================================================
  SerialDebug.println();
  SerialDebug.println("ET-ESP32-RS485...Demo Modbus RTU RS485");
  SerialDebug.println("Interface Modbus RTU Relay2");
  //===============================================================================================

  //===============================================================================================
  // Mosbus RTU Command
  //=============================================================================================== 
  //au16dataRelay[0] = 0x0300;                                                                      // Toggle Output Command
  au16dataRelay[0] = 0x0400;                                                                      // Latch Output Command
  //au16dataRelay[0] = 0x0500;                                                                      // Momentary Output Command
  //au16dataRelay[0] = 0x0600;                                                                      // Delaye Output Command
  //===============================================================================================
  // telegram[0]: Toggle Relay1 device[1]
  //            : Modbus Relay RS485(Modbus RTU)
  //            : Device ID[1]
  // Device Addr,Function Code(06),MSB Reg Addr,LSB Reg Addr,MSB Value,LSB Value,CRC16
  //===============================================================================================
  telegram[0].u8id = 1;                                                                           // slave address = 1
  telegram[0].u8fct = 6;                                                                          // function code (toggle relay1)
  telegram[0].u16RegAdd = 0x0001;                                                                 // start address in slave
  telegram[0].u16CoilsNo = 1;                                                                     // number of elements (coils or registers) to read
  telegram[0].au16reg = au16dataRelay;                                                            // pointer to a memory array in the Arduino
  //===============================================================================================
  // telegram[1]: Toggle Relay2 device[1]
  //            : Modbus Relay RS485(Modbus RTU)
  //            : Device ID[1]
  // Device Addr,Function Code(06),MSB Reg Addr,LSB Reg Addr,MSB Value,LSB Value,CRC16
  //===============================================================================================
  telegram[1].u8id = 1;                                                                          // slave address = 1
  telegram[1].u8fct = 6;                                                                         // function code (this one is registers read)
  telegram[1].u16RegAdd = 0x0002;                                                                // start address in slave
  telegram[1].u16CoilsNo = 1;                                                                    // number of elements (coils or registers) to read
  telegram[1].au16reg = au16dataRelay;                                                           // pointer to a memory array in the Arduino
  //===============================================================================================
  master.begin(SerialRS485);                                                                     // Mosbus Interface
  master.setTimeOut(2000);                                                                       // if there is no answer in 2000 ms, roll over
  u32wait = millis() + 2000;
  u8state = u8query = 0; 
  //===============================================================================================
}

//=================================================================================================
// Start of Test : MODBUS RTU RELAY2
//=================================================================================================
void loop() 
{
  switch( u8state ) 
  {
    case 0: 
      if (millis() > u32wait) u8state++;        // wait state
    break;
    
    case 1: 
      master.query(telegram[u8query]);          // send query (only once)
      u8state++;
      u8query++;
      if(u8query > 2) u8query = 0;              // 0,1,...,0
    break;

    
    case 2:
      master.poll();                            // check incoming messages
      if(master.getState() == COM_IDLE) 
      {
        u8state = 0;
        u32wait = millis() + 2000; 
      } 
    break;
  }
}
//=================================================================================================
// End of Test : MODBUS RTU RELAY2
//=================================================================================================

